/*
 *  Ce fichier fait partie intgrante du projet TuringQTwin32
 *
 *  auteur: Olivier Gies - olivier.gies@aist.enst.fr
 *
 *  Tous droits rservs
 */
#include "grid.h"

Grid::Grid( int w,int h, double v, int cs,int cw, int x,int y, Color3 c )
: m_height( h ), m_width( w ),
	m_nCellSpacing( cs ), m_nCellWidth( cw ),
	m_nX( x ), m_nY( y ),
	m_mainColor( c )
{
	m_data = new double[m_width*m_height];

	for( int i=0; i<w*h; i++ )
		m_data[i] = v;

	m_palSize = 256;
	m_pal = NULL;
	spreadPalette();
}

Grid::Grid( const Grid &g )
: m_height( g.getHeight() ), m_width( g.getWidth() ),
	m_palSize( g.getPalSize() ),
	m_nCellSpacing( g.getCellSpacing() ), m_nCellWidth( g.getCellWidth() ),
	m_nX( g.getX() ), m_nY( g.getY() ),
	m_mainColor( g.getMainColor() )
{
	m_data = new double[m_width*m_height];
	memcpy( m_data, g.m_data, m_width*m_height*sizeof(double) );
//	for( int i=0; i<m_width*m_height; i++ )
//		m_data[i] = g.m_data[i];

	m_pal = NULL;
	takePaletteFrom( &g );
}

Grid::~Grid()
{
	delete [] m_data;
	delete [] m_pal;
}

Grid Grid::operator =( Grid n )
{
	m_width = n.m_width;
	m_height = n.m_height;
	delete [] m_data;
	m_data = new double[m_width*m_height];
	for( int i=0; i<m_width*m_height; i++ )
		m_data[i] = n.m_data[i];

	takePaletteFrom( &n );

	return *this;
}

Grid operator +( Grid a, Grid b )
	// WARNING: Palette parameters are taken from left operand ! (here 'a')
{
	assert(
		a.getWidth() == b.getWidth() &&
		a.getHeight() == b.getHeight() &&
		a.getPalSize() == b.getPalSize() );

	Grid c( a.getWidth(), a.getHeight() );
	double* ad = a.getData();
	double* bd = b.getData();
	double* cd = c.getData();
	for( int i=0; i<c.getWidth()*c.getHeight(); i++ )
	{
		cd[i] = ad[i] + bd[i];
	}

	c.takePaletteFrom( &a );

	return c;
}

Grid Grid::operator +=( Grid g )
{
	assert(
		getWidth() == g.getWidth() &&
		getHeight() == g.getHeight() );

	double* gdata = g.getData();

	for( int i=0; i<m_width*m_height; i++ )
		m_data[i] += gdata[i];

	return *this;
}

double *Grid::getData() {
	return m_data;
}

bool Grid::setValue( int i, int j, double v )
{
	assert( ( i<m_width ) && ( j<m_height ) );

	m_data[i+m_width*j] = v;
	return true;
}

bool Grid::addValue( int i, int j, double v )
{
	assert( ( i<m_width ) && ( j<m_height ) );
	m_data[i+m_width*j] += v;
	return true;
}

double Grid::getValue( int i, int j )
{
	assert( ( i<m_width ) && ( j<m_height ) );
	return m_data[i+m_width*j];
}

int Grid::getWidth() const {
	return m_width;
}

int Grid::getHeight() const {
	return m_height;
}

int Grid::getPalSize() const {
	return m_palSize;
}

void Grid::setCellSpacing( int n ) {
	m_nCellSpacing = n;
}

void Grid::setCellWidth( int n ) {
	m_nCellWidth = n;
}

void Grid::setX( int n ) {
	m_nX = n;
}

void Grid::setY( int n ) {
	m_nY = n;
}

int Grid::getCellSpacing() const
{
	return m_nCellSpacing;
}

int Grid::getCellWidth() const
{
	return m_nCellWidth;
}

int Grid::getX() const
{
	return m_nX;
}

int Grid::getY() const
{
	return m_nY;
}

void Grid::setMainColor( Color3 c )
{
	m_mainColor = c;
	spreadPalette();
}

void Grid::setMainColor( int r, int g, int b )
{
	setMainColor( Color3( r, g, b ) );
}

Color3 Grid::getMainColor() const
{
	return m_mainColor;
}

Color3* Grid::getPalette() const
{
	return m_pal;
}

void Grid::spreadPalette()
{
	if( m_pal )
		delete [] m_pal;

	m_pal = new Color3[m_palSize];
	for( int j=0; j<m_palSize; j++ )
	{
		m_pal[j].m_R = m_mainColor.m_R * ( ((float)j)/((float)m_palSize) );
		m_pal[j].m_G = m_mainColor.m_G * ( ((float)j)/((float)m_palSize) );
		m_pal[j].m_B = m_mainColor.m_B * ( ((float)j)/((float)m_palSize) );
	}
}

void Grid::takePaletteFrom( const Grid *g )
{
	m_palSize = g->getPalSize();
	if( m_pal )
		delete [] m_pal;
	m_pal = new Color3[m_palSize];
	Color3* gpal = g->getPalette();
//	for( int j=0; j<m_palSize; j++ )
//		m_pal[j] = gpal[j];
	memcpy( m_pal, gpal, m_palSize*sizeof( Color3 ) );
}

void Grid::print()
{
	for( int i=0; i<m_width; i++ )
	{
		for( int j=0; j<m_height; j++ )
		{
			printf( "%f\t", getValue( i, j ) );
		}
		printf( "\n" );
	}
}

void Grid::show( IDisplay *d )
{
	assert( d );

	for( int i=0; i<m_width; i++ )
	{
		for( int j=0; j<m_height; j++ )
		{
			double v = getValue( i, j );
			if( v<0 )
				v = 0;
			if( v>1 )
				v = 1;
			
			//assert ( ( v>=0 ) && ( v <= 1 ) );
			Color3 c = m_pal[(int)(v*255)];

			int x = m_nX + m_nCellSpacing + i * (m_nCellSpacing + m_nCellWidth);
			int y = m_nY + m_nCellSpacing + j * (m_nCellSpacing + m_nCellWidth);

			for( int k=0; k<m_nCellWidth; k++ )
				for( int l=0; l<m_nCellWidth; l++ )
					d->drawPixel( x+k, y+l, c );
		}
	}
}

double Grid::laplacian( int i, int j, int mode )
{
	assert( ( j>=0 ) && ( j<m_height ) );
	assert( ( i>=0 ) && ( i<m_width ) );

	double left, right, top, bottom, center;

	center = getValue( i, j );

	int ii, jj;

	switch( mode )
	{
	case 3: // full torus
		ii = (i-1+m_width)%m_width;
		left = getValue( ii, j );
		right = getValue( (i+1)%m_width, j );
		jj = (j-1+m_height)%m_height;
		top = getValue( i, jj );
		bottom = getValue( i, (j+1)%m_height );
		break;
	case 2: // horizontal cylinder
		ii = (i-1+m_width)%m_width;
		left = getValue( ii, j );
		right = getValue( (i+1)%m_width, j );
		if( j == 0 )
			top = 0;
		else
			top = getValue( i, j-1 );

		if( j == m_height-1 )
			bottom = 0;
		else
			bottom = getValue( i, j+1 );
		break;
	case 1: // vertical cylinder
		if( i == 0 )
			left = 0;
		else
			left = getValue( i-1, j );

		if( i == m_width-1 )
			right = 0;
		else
			right = getValue( i+1, j );
		jj = (j-1+m_height)%m_height;
		top = getValue( i, jj );
		bottom = getValue( i, (j+1)%m_height );
		break;
	case 0: // rectangle
	default:
		if( i == 0 )
			left = 0;
		else
			left = getValue( i-1, j );

		if( i == m_width-1 )
			right = 0;
		else
			right = getValue( i+1, j );

		if( j == 0 )
			top = 0;
		else
			top = getValue( i, j-1 );

		if( j == m_height-1 )
			bottom = 0;
		else
			bottom = getValue( i, j+1 );
		break;
	}

	double lap = left + right + top + bottom - 4*center;

	return lap;
}