/*
 *  Ce fichier fait partie intgrante du projet TuringQTwin32
 *
 *  auteur: Olivier Gies - olivier.gies@aist.enst.fr
 *
 *  Tous droits rservs
 */
#ifndef GRID_H
#define GRID_H

#include "IDisplay.h"

#include "color3.h"

#include <iostream>
#include <cassert>

/*
 * The Grid class represents a matrix of cells.
 * It is used to model the concentrations of a morphogene on the whole
 * reaction area.
 */

class Grid
{
public:
	Grid( int = 1, int = 1, double = 0.0, int = 1, int = 5, int = 10, int = 10, Color3 = Color3( 255, 0, 0 ) );
	// copy constructor
	Grid( const Grid & );
	~Grid();

	Grid operator =( Grid g );
	// only the data is added ; color palette is left unchanged
	Grid operator +=( Grid g );

	// Read acces to dimensions
	int getHeight() const;
	int getWidth() const;

	// Data management
	double *getData();
	bool setValue( int i, int j, double v );
	bool addValue( int i, int j, double v );
	double getValue( int i, int j );

	// R/W access to visualization parameters
	void setCellSpacing( int );
	void setCellWidth( int );
	void setX( int );
	void setY( int );
	int getCellSpacing() const;
	int getCellWidth() const;
	int getX() const;
	int getY() const;

	// The main color parametrizes the color palette
	void setMainColor( Color3 );
	void setMainColor( int, int, int );
	Color3 getMainColor() const;

	// Color palette management
	// The color palette is faded from black to m_mainColor
	int getPalSize() const;
	void spreadPalette();
	Color3 *getPalette() const; // warning: pointed content may change
	void takePaletteFrom( const Grid * );

	// Render the Grid onto a given IDisplay
	//   nb: this procedure could be taken the other way
	//       cf. idisplay.h for additionnal comments
	void show( IDisplay *d );

	// echoes the Grid to the standard output (stdout.txt on win32 system)
	void print();

	// computes a discrete laplacian for cell at position (i,j),
	// according to the torus mode
	double laplacian( int i, int j, int mode );

private:
	// Dimensions
	int m_height, m_width;
	// Data
	double *m_data;

	// Color varaibles
	Color3 m_mainColor;
	int m_palSize;
	Color3 *m_pal;

	// render parameters (square cells)
	int m_nCellSpacing; // also makes a border around the whole grid
	int m_nCellWidth; // = height because of squareness
	int m_nX, m_nY; // coordinates of the Grid (upper-left corner) within the display window
};

// DO NOT USE THIS OPERATOR YET
Grid operator +( Grid a, Grid b );

#endif