/*
 *  Ce fichier fait partie intgrante du projet TuringQTwin32
 *
 *  auteur: Olivier Gies - olivier.gies@aist.enst.fr
 *
 *  Tous droits rservs
 */
#include "reactor.h"

#include <cstdlib>
#include <SDL/SDL.h>

Reactor::Reactor( int w, int h, IDisplay *d ) : m_display( d )
{
	// parametres
	m_A = Grid( w, h, 1.0 );
	m_B = Grid( w, h, 0.0 );

	srand( SDL_GetTicks() );

	m_currentStep = 0;
	m_showStep = 10;
	m_stopStep = -1;

	// affichage
	m_bgColor = Color3( 0, 0, 0 );

	m_A.setMainColor( 255, 192, 0 );
	m_B.setMainColor( 0, 127, 255 );

	setVisibleA( true );
	setVisibleB( false );

	m_A.setX( 10 );
	m_A.setY( 10 );

	m_B.setX( m_A.getX() + (m_A.getCellSpacing() + m_A.getCellWidth())*m_A.getWidth() + 20 );
	m_B.setY( 10 );

	m_Da = 0.1;
	m_Db = 0.05;
	m_torusMode = 3;

	resetGrids();
}

Reactor::~Reactor()
{
}

void Reactor::show()
{
	if( !m_display )
		return;

	m_display->clearScreen( m_bgColor );
	if( isVisibleA() )
		m_A.show( m_display );
	if( isVisibleB() )
		m_B.show( m_display );
	m_display->update();
}

void Reactor::showA()
{
	m_A.show( m_display );
}

void Reactor::showB()
{
	m_B.show( m_display );
}

void Reactor::setDisplay( IDisplay *d )
{
	m_display = d;
}

void Reactor::setWidth( int w )
{
	setDimensions( w, getHeight() );
}

void Reactor::setHeight( int h )
{
	setDimensions( getWidth(), h );
}

void Reactor::setDimensions( int w, int h )
{
	Color3 ca = m_A.getMainColor(), cb = m_B.getMainColor();
	m_A = Grid( w, h ); m_A.setMainColor( ca );
	m_B = Grid( w, h ); m_B.setMainColor( cb );
	m_B.setX( m_A.getX() + ( m_A.getCellSpacing() + m_A.getCellWidth())*m_A.getWidth() + 20 );
	resetGrids();
}

Grid *Reactor::getAPtr()
{
	return &m_A;
}

Grid *Reactor::getBPtr()
{
	return &m_B;
}

int Reactor::getWidth()
{
	return m_A.getWidth();
}

int Reactor::getHeight()
{
	return m_A.getHeight();
}

IDisplay *Reactor::getDisplay()
{
	return m_display;
}

void Reactor::update( double t )
{
	int w = m_A.getWidth();
	int h = m_A.getHeight();

	Grid new_A( w, h );
	Grid new_B( w, h );

	for( int i=0; i<w; i++ )
		for( int j=0; j<h; j++ )
		{
			double vA, vB;

			vA = m_Da * m_A.laplacian( i, j, m_torusMode ) + funcA( i, j );
			vB = m_Db * m_B.laplacian( i, j, m_torusMode ) + funcB( i, j );

			new_A.setValue( i,j, vA );
			new_B.setValue( i,j, vB );
		}

	m_A += new_A;
	m_B += new_B;
	if( !( m_currentStep%m_showStep ) )
		show();
	incrementStep();
//	m_A.print();
}

void Reactor::initReaction( int W, int H )
{
	m_A = Grid( W, H, 1.0 );
	m_B = Grid( W, H, 0.0 );

	m_A.setMainColor( 255, 223, 63 );
	m_B.setMainColor( 127, 127, 255 );

	int wint = 6, hint = 6;
	for( int i=-wint/2; i<wint/2; i++ )
		for( int j=-hint/2; j<hint/2; j++ )
		{
			m_A.setValue( W/4+i, 3*H/5+j, 0.5 );
			m_B.setValue( W/4+i, 3*H/5+j, 0.25 );
		}
}

bool Reactor::isVisibleA()
{
	return m_isVisibleA;
}

void Reactor::setVisibleA( bool enable )
{
	m_isVisibleA = enable;
}

bool Reactor::isVisibleB()
{
	return m_isVisibleB;
}

void Reactor::setVisibleB( bool enable )
{
	m_isVisibleB = enable;
}

void Reactor::resetGrids()
{
	double dRM = (double)RAND_MAX;
	double a, b;
	double ia = 0.1, ib = 0.5;
	double id = 0.1;

	int w = getWidth();
	int h = getHeight();

	int i, j;

	for( int k = 0; k<w; k++ )
	for( int l = 0; l<h; l++ )
	{
		m_A.setValue( k, l, 1.0 );
		m_B.setValue( k, l, 0.0 );
	}

	int wint = 10, hint = 10;
	for( i=-wint/2; i<wint/2; i++ )
	for( j=-hint/2; j<hint/2; j++ )
//	for( int i=0; i<getWidth(); i++ )
//	for( int j=0; j<getHeight(); j++ )
		{
			a = ia + id*((double)(rand()-RAND_MAX))/dRM;
			b = ib + id*((double)(rand()-RAND_MAX))/dRM;
			m_A.setValue( w/2+i, 3*h/4+j, a );
			m_B.setValue( w/2+i, 3*h/4+j, b );
//			m_A.setValue( i, j, a );
//			m_B.setValue( i, j, b );
		}
	for( i=-wint/2; i<wint/2; i++ )
	for( j=-hint/2; j<hint/2; j++ )
//	for( int i=0; i<getWidth(); i++ )
//	for( int j=0; j<getHeight(); j++ )
		{
			a = ia + id*((double)(rand()-RAND_MAX))/dRM;
			b = ib + id*((double)(rand()-RAND_MAX))/dRM;
			m_A.setValue( w/4+i, h/4+j, a );
			m_B.setValue( w/4+i, h/4+j, b );
//			m_A.setValue( i, j, a );
//			m_B.setValue( i, j, b );
		}
/*	for( i=-wint/2; i<wint/2; i++ )
	for( j=-hint/2; j<hint/2; j++ )
//	for( int i=0; i<getWidth(); i++ )
//	for( int j=0; j<getHeight(); j++ )
		{
			a = ia + id*((double)(rand()-RAND_MAX))/dRM;
			b = ib + id*((double)(rand()-RAND_MAX))/dRM;
			m_A.setValue( w/4+i, h/2+j, a );
			m_B.setValue( w/4+i, h/2+j, b );
//			m_A.setValue( i, j, a );
//			m_B.setValue( i, j, b );
		}
	for( i=-wint/2; i<wint/2; i++ )
	for( j=-hint/2; j<hint/2; j++ )
//	for( int i=0; i<getWidth(); i++ )
//	for( int j=0; j<getHeight(); j++ )
		{
			a = ia + id*((double)(rand()-RAND_MAX))/dRM;
			b = ib + id*((double)(rand()-RAND_MAX))/dRM;
			m_A.setValue( 3*w/4+i, h/2+j, a );
			m_B.setValue( 3*w/4+i, h/2+j, b );
//			m_A.setValue( i, j, a );
//			m_B.setValue( i, j, b );
		}*/
}

void Reactor::setCellWidth( int n )
{
	m_A.setCellWidth( n );
	m_B.setCellWidth( n );
	m_B.setX( m_A.getX() + (m_A.getCellSpacing() + m_A.getCellWidth())*m_A.getWidth() + 20 );
}

int Reactor::getCellWidth()
{
	return m_A.getCellWidth();
}

void Reactor::setCellSpacing( int n )
{
	m_A.setCellSpacing( n );
	m_B.setCellSpacing( n );
	m_B.setX( m_A.getX() + (m_A.getCellSpacing() + m_A.getCellWidth())*m_A.getWidth() + 20 );
}

int Reactor::getCellSpacing()
{
	return m_A.getCellSpacing();
}

void Reactor::setASpreadRate( double a )
{
	m_Da = a;
}

void Reactor::setBSpreadRate( double b )
{
	m_Db = b;
}

double Reactor::getASpreadRate()
{
	return m_Da;
}

double Reactor::getBSpreadRate()
{
	return m_Db;
}

void Reactor::setTorusMode( int t )
{
	m_torusMode = t;
}

int Reactor::getTorusMode()
{
	return m_torusMode;
}

int Reactor::incrementStep()
{
	m_currentStep++;
	if( m_currentStep == m_stopStep )
		stop();
	return m_currentStep;
}

void Reactor::setCurrentStep( unsigned int n )
{
	m_currentStep = n;
}

int Reactor::getCurrentStep()
{
	return m_currentStep;
}

void Reactor::setShowStep( unsigned int n )
{
	m_showStep = n;
}

int Reactor::getShowStep()
{
	return m_showStep;
}

void Reactor::setStopStep( int n )
{
	m_stopStep = n;
}