/*
 *  Ce fichier fait partie intgrante du projet TuringQTwin32
 *
 *  auteur: Olivier Gies - olivier.gies@aist.enst.fr
 *
 *  Tous droits rservs
 */
#ifndef REACTOR_H
#define REACTOR_H

#include "idisplay.h"
#include "grid.h"

/*
 * The Reactor class accounts for the reaction area.
 * It contains 2 Grid objects accounting for the morphogenes
 * concentrations.
 *
 * It is a pure abstract class via the private functions funcA() and funcB(),
 * which account for the cross-reaction components of the differential
 * equations in the reaction-diffusion system, and resetGrids() which sets up
 * the initial state.
 */

class Reactor
{
public:
	Reactor( int w, int h, IDisplay *d );
	~Reactor();

	// Initializes various parameters
	void initReaction( int, int );

	// Give acces to activator and inhibitor Grids
	Grid *getAPtr();
	Grid *getBPtr();

	// R/W access to spread rates
	void setASpreadRate( double );
	void setBSpreadRate( double );
	double getASpreadRate();
	double getBSpreadRate();

	// R/W access to torus mode
	void setTorusMode( int );
	int getTorusMode();

	// R/W access to reactor dimensions
	void setWidth( int );
	void setHeight( int );
	void setDimensions( int width, int height );
	int getWidth();
	int getHeight();

	// Performs a reaction step
	//   nb: parameter t is not used so far, but meant to allow
	//       time control on the reaction
	void update( double t );

	// Step-counter management
	int incrementStep();
	void setCurrentStep( unsigned int );
	int getCurrentStep();
	void setShowStep( unsigned int );
	int getShowStep();
	void setStopStep( int );

	// Initializes the reaction
	virtual void resetGrids();

	// R/W to the display object onto which the Reactor will be drawn
	void setDisplay( IDisplay *s );
	IDisplay *getDisplay();

	// Visibility of morphogenes
	bool isVisibleA();
	void setVisibleA( bool enable );
	bool isVisibleB();
	void setVisibleB( bool enable );

	// Visual parameters for grids
	void setCellWidth( int );
	int getCellWidth();
	void setCellSpacing( int );
	int getCellSpacing();

	// Renders the reactor onto its m_display member
	void show();

	// obsolete
	void showA();
	void showB();

	// Implemented in QReactor, but needed in incrementStep()
	virtual void stop() = 0;

protected:
	// Morphogenes concentrations
	Grid m_A, m_B;
	// Morphogenes spread rates
	double m_Da, m_Db;
	// Torus mode : 3 is full torus, 2 is horizontal, 1 is vertical and 0 is none
	int m_torusMode;

	// Step management
	int m_currentStep, m_showStep;
	int m_stopStep;

	// Cross-reaction components
	virtual double funcA( int i, int j ) = 0;
	virtual double funcB( int i, int j ) = 0;

	// Display object
	IDisplay *m_display;
	// Backrgound color
	Color3 m_bgColor;

	bool m_isVisibleA, m_isVisibleB;
};

#endif