/*
 *  Ce fichier fait partie intgrante du projet TuringQTwin32
 *
 *  auteur: Olivier Gies - olivier.gies@aist.enst.fr
 *
 *  Tous droits rservs
 */
#include "sdldisplay.h"

SDLDisplay *SDLDisplay::pinstance = NULL;

SDLDisplay::SDLDisplay()
{
	isLocked = false;
	xRes = 640;
	yRes = 480;

	initSDL();
}

SDLDisplay::~SDLDisplay()
{
	SDL_Quit();
}

SDLDisplay *SDLDisplay::instance()
{
	if( pinstance == NULL )
		pinstance = new SDLDisplay();
	return pinstance;
}

void SDLDisplay::clearScreen( Color3 c )
{
	clearScreen( c.m_R,c.m_G,c.m_B );
}

void SDLDisplay::clearScreen( int r, int g, int b )
{
	SDL_FillRect( m_screen, NULL, SDL_MapRGB( m_screen->format, r,g,b ) );
}

void SDLDisplay::drawPixel( int x, int y, Color3 c )
{
	if( !isLocked )
		lock();

	if( !( x<xRes && x>-1 && y<yRes && y>-1 ) ) // draws only inside the screen
		return;

	Uint32 color = SDL_MapRGB( m_screen->format, c.m_R,c.m_G,c.m_B );
	switch( m_screen->format->BytesPerPixel )
	{
	case 1: // Assuming 8-bpp
		{
			Uint8 *bufp;
			bufp = (Uint8 *)m_screen->pixels + y*m_screen->pitch + x;
			*bufp = color;
		}
		break;
	case 2: // Probably 15-bpp or 16-bpp
		{
			Uint16 *bufp;
			bufp = (Uint16 *)m_screen->pixels + y*m_screen->pitch/2 + x;
			*bufp = color;
		}
		break;
	case 3: // Slow 24-bpp mode, usually not used
		{
			Uint8 *bufp;
			bufp = (Uint8 *)m_screen->pixels + y*m_screen->pitch + x * 3;
			if(SDL_BYTEORDER == SDL_LIL_ENDIAN)
			{
				bufp[0] = color;
				bufp[1] = color >> 8;
				bufp[2] = color >> 16;
			} else {
				bufp[2] = color;
				bufp[1] = color >> 8;
				bufp[0] = color >> 16;
			}
		}
		break;
	case 4: // Probably 32-bpp
		{
			Uint32 *bufp;
			bufp = (Uint32 *)m_screen->pixels + y*m_screen->pitch/4 + x;
			*bufp = color;
		}
		break;
	}
}

void SDLDisplay::update()
{
	if( isLocked )
		unlock();

	SDL_Flip( m_screen );
}

/*
void SDLDisplay::changeBackground()
{
	Color3 c( 255, 127, 0 );

	for( int i=0; i<640; i++ )
		for( int j=0; j<480; j++ )
		{
			this->drawPixel( i, j, c );
		}

	this->update();
}
*/

void SDLDisplay::lock()
{
	if ( SDL_MUSTLOCK( m_screen ) )
	{
		if ( SDL_LockSurface( m_screen ) < 0 )
		{
			return;
		}
	}
	isLocked = true;
}

void SDLDisplay::unlock()
{
	if ( SDL_MUSTLOCK( m_screen ) )
	{
		SDL_UnlockSurface( m_screen );
	}
	isLocked = false;
}

bool SDLDisplay::initSDL()
{
	bool r = true;

	if ( SDL_Init( SDL_INIT_AUDIO|SDL_INIT_VIDEO ) < 0 )
	{
		printf( "Unable to init SDL: %s\n", SDL_GetError() );
		r = false;
	}

	m_screen = SDL_SetVideoMode( 640,480, 32, SDL_HWSURFACE|SDL_DOUBLEBUF );
	if ( m_screen == NULL )
	{
		printf( "Unable to set 640x480 (32bpp) video mode: %s\n", SDL_GetError() );
		r = false;
	}

	return r;
}